/************************************************************************
 *
 * \file: EventDispatcher.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * <brief description>.
 * <detailed description>
 * \component: Android Auto - demo
 *
 * \author: Sonali Pujari / SonaliSajan.Pujari@in.bosch.com
 *
 * \copyright (c) 2017 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/
#include <adit_logging.h>
#include "EventDispatcher.h"

LOG_IMPORT_CONTEXT(demo)

namespace adit { namespace aauto {

EventDispatcher::EventDispatcher()
{
    mEventDispatcherThreadId = -1;
    mRunning    = false;
    mSemCreated = false;
}

EventDispatcher::~EventDispatcher()
{
    LOG_INFO((demo, "EventDispatcher::%s()  called", __func__));
    stopEventDispatcher();
}

bool EventDispatcher::startEventDispatcher()
{
    bool res = false;
    mRunning = true;

    int32_t err = sem_init(&mEventAvailable, 0, 0);
    if (err == -1) {
        LOG_ERROR((demo, "EventDispatcher::%s()  create Semaphore failed err=%d, errno=%d",
                __func__, err, errno));
        res = false;
    } else {
        mSemCreated = true;
    }

    /* create thread */
    err = pthread_create(&mEventDispatcherThreadId, nullptr, &EventDispatcherThread, this);
    if (0 != err) {
        LOG_ERROR((demo, "EventDispatcher::%s()  create EventDispatcherThread failed err=%d, errno=%d",
                __func__, err, errno));
        res = false;
    } else {
        LOG_INFO((demo, "EventDispatcher::%s()  EventDispatcherThread created", __func__));
        res = true;
    }
    return res;
}

void EventDispatcher::stopEventDispatcher()
{
    bool wasRunning = mRunning;
    mRunning = false;

    if (false == wasRunning) {
        LOG_INFO((demo, "EventDispatcher::%s()  already stopped", __func__));
        return;
    }

    if (true == mSemCreated) {
        /* this should wake-up the EventHandlerThread */
        sem_post(&mEventAvailable);
    }

    if (mEventDispatcherThreadId > 0) {
        int32_t err = pthread_join(mEventDispatcherThreadId, nullptr);
        LOG_INFO((demo, "EventDispatcher::%s()  join EventDispatcherThread returned with err=%d, errno=%d",
                        __func__, err, errno));
        mEventDispatcherThreadId = -1;
    }

    if (true == mSemCreated) {
        sem_destroy(&mEventAvailable);
        LOG_INFO((demo, "EventDispatcher::%s()  sem_destroy done", __func__));
        mSemCreated = false;
    }
}

void* EventDispatcher::EventDispatcherThread(void* context)
{
    auto me = static_cast<EventDispatcher*>(context);
    if (me == nullptr) {
        LOG_ERROR((demo, "EventDispatcher::%s()  could not cast input pointer", __func__));
        return nullptr;
    }

    while (me->mRunning)
    {
        std::shared_ptr<DevEventItem> event = me->dequeueEvent();
        if (false == me->mRunning) {
            LOG_INFO((demo, "EventDispatcher::%s()  mRunning was set to false ", __func__));
            // Happens while shutting down.
            break;
        } else if (event == nullptr) {
            LOG_WARN((demo, "EventDispatcher::%s()  event is NULL", __func__));
            // Maybe happens while shutting down.
            break;
        }

        /* handle event */
        me->handleEvent(event);
    }

    LOG_INFO((demo, "EventDispatcher::%s()  EventDispatcher stopped", __func__));

    return nullptr;
}

void EventDispatcher::handleEvent(std::shared_ptr<DevEventItem> inEventItem)
{
    DevEventType eventType = inEventItem->getEventType();
    std::string serial = inEventItem->getSerial();

    if (DevEventType::ByeByeRequest == eventType) {
        LOG_INFO((demo, "EventDispatcher::%s() onByeByeRequest ", __func__));
        onByeByeRequest(serial);
    } else if (DevEventType::OTHER == eventType) {
        LOG_INFO((demo, "EventDispatcher::%s() onOtherEvent ", __func__));
    } else {
        LOG_ERROR((demo, "EventDispatcher::%s()  unknown event %d",
                __func__, (int32_t)eventType));
    }
}

void EventDispatcher::queueEvent(const std::shared_ptr<DevEventItem>& work)
{
    if (mRunning)
    {
        {
            std::lock_guard<std::mutex> lock(mEventQueueMutex);

            mEventQueue.push_back(work);

            LOG_INFO((demo, "EventDispatcher::%s()  No. of events in list:  %zd", __func__, mEventQueue.size()));
        } /*mEventQueueMutex is automatically  released here when lock goes out of scope*/

        sem_post(&mEventAvailable);
    }
    else
    {
        LOG_INFO((demo, "EventDispatcher::%s()  mRunning was set, event discarded", __func__));
    }
}

std::shared_ptr<DevEventItem> EventDispatcher::dequeueEvent()
{
    if (mRunning) {
        sem_wait(&mEventAvailable);
        LOG_INFO((demo, "EventDispatcher::%s()  sem_wait released", __func__));
    } else {
        LOG_INFO((demo, "EventDispatcher::%s()  mRunning was set, ignore sem_wait", __func__));
    }

    std::shared_ptr<DevEventItem> event(nullptr);
    {
        std::lock_guard<std::mutex> lock(mEventQueueMutex);

        /* lock access to queue here! */
        if (mEventQueue.size() == 0) {
            LOG_INFO((demo, "EventDispatcher::%s()  mEventQueue is empty!", __func__));
            return nullptr;
        }
        event = mEventQueue.front();
        mEventQueue.pop_front();
    }/*mEventQueueMutex is automatically  released here when lock goes out of scope*/

    return event;
}

} } // namespace adit { namespace aauto {



